function h = minus(f, g)
%MINUS   implements - taking into account cancellation.
%   MINUS implements f - g for flaps.  Because cancellation can promote
%   rounding error in f.d and g.d where it can be seen at FlapRlevel,
%   it is necessary to compute a smaller rounding level.  The process
%   is expensive, requiring and extra log of f-g.  When FlapRleve < 8,
%   this problem does not occur.


global FlapRlevel

f = flap(f);
g = flap(g);
if FlapRlevel<8 | FlapRlevel==15 | FlapRlevel==16

  % Cancellation cannot cause problems in this range.
  % Compute the difference and round.

   h = flap(f.d - g.d);
else

   % Cancellation can cause rounding error from the basement
   % to rise above FlapRlevel.  When this occurs, round at a
   % lower level.

   hd = f.d - g.d;
   if isreal(hd)   
      denom = max(abs(f.d),abs(g.d));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    FlapRlevel + ceil(1+log10((abs(hd)+(hd==0))./denom)));
      hd = FlapRound(hd, Rlevel);
      h = flap;
      h.Rlevel = FlapRlevel;
      h.d = hd;
   else
      hrd = real(hd);
      denom = max(abs(real(f.d)),  abs(real(g.d)));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    FlapRlevel + ceil(1+log10((abs(hrd)+(hrd==0))./denom)));
      hrd = FlapRound(hrd, Rlevel);
      hid = imag(hd);
      denom = max(abs(imag(f.d)),  abs(imag(g.d)));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    FlapRlevel + ceil(1+log10((abs(hid)+(hid==0))./denom)));
      hid = FlapRound(hid, Rlevel);
      h = flap;
      h.Rlevel = FlapRlevel;
      h.d = complex(hrd, hid);
   end
end
