function H = mldivide(F, G)
%MLDIVIDE  implements the \
%   H = MLDIVIDE(F, G) computes H = INV(L)*G.   L must be square.
%   In general, the function computes an LU factorization of H
%   (see LUD) and solves triangular systems to get the H.
%   However, if F is triangular, it solves the system directly
%   without computing a factorization.  The argument F may be a scalar.


F = flap(F);
G = flap(G);
[m, n] = size(F.d);

if m ~= n
   error('Flap: In the expression F\G, the flap F must be square.')
end

% If F is a scalar, divide and get out.

if m == 1
   H = flap(F.d\G.d);
   return;
end

[nn, p] = size(G.d);

if nn ~= n
   error('Flap: Dimensions inconsistent for division.')
end

H = G.d;

if tril(F.d, -1) == 0

%  F is upper triangular.

   F = F.d;
   if ~all(diag(F))
      error('Flap: The divisor is singular.');
   end

   for i=m:-1:1
      H(i,:) = FlapRound(H(i,:)/F(i,i));
      H(1:i-1,:) = FlapRound(H(1:i-1,:) - FlapRound(F(1:i-1,i)*H(i,:)));
   end

elseif triu(F.d, 1) == 0

%  F is lower triangular

   F = F.d;
   if ~all(diag(F))
      error('Flap: The divisor is singular.');
   end

   for i=1:m
      H(i,:) = FlapRound(H(i,:)/F(i,i));
      H(i+1:m,:) = FlapRound(H(i+1:m,:) - FlapRound(F(i+1:m,i)*H(i,:)));
   end

else

%  General matrix.  Compute the LU decomposition of F
%  and solve the systems.

   [L, U, P] = lud(F);
   L = L.d;
   U = U.d;


   H = P.d*G.d;
   
   for i=1:m-1
      H(i+1:m,:) = FlapRound(H(i+1:m,:) - FlapRound(L(i+1:m,i)*H(i,:)));
   end

   for i=m:-1:1
      H(i,:) = FlapRound(H(i,:)/U(i,i));
      H(1:i-1,:) = FlapRound(H(1:i-1,:) - FlapRound(U(1:i-1,i)*H(i,:)));
   end
end
H = flap(H);
