function [Q, R, E] = qrd(A, econ)
%QRD   Computes a QR decomposition.
%   [Q,R] = QRD(A) produces an upper triangular matrix R of the same
%   dimension as A and a unitary matrix Q so that A = Q*R.
%   In this and the cases below, Q and R are normalized so that
%   the diagonal elements of R are nonnegative.
%
%   [Q,R,E] = QRD(A) produces a permutation matrix E, an upper
%   triangular R and a unitary Q so that A*E = Q*R.  The column
%   permutation E is chosen so that 
%
%      R(k,k) >= norm(R(k:end, j)),  j>k.                 (*)
%
%   [Q,R] = QRD(A,0) produces the "economy size" decomposition.
%   If A is m-by-n with m > n, then only the first n columns of Q
%   are computed.
%
%   [Q,R,E] = QRD(A,0) produces an "economy size" decomposition in
%   which E is a permutation vector, so that Q*R = A(:,E).  The column
%   permutation E is chosen so that (*) is satisfied.
%


A = flap(A);
[m, n] = size(A);

% Set up economy size Q.

if nargin==2
   if econ ~= 0
      error('Use qr(X,0) for economy size decomposition.')
   else
      econ = true;
   end
else
   econ = false;
end

% Set up pivoting.

if nargout == 3
   pivot = true;
   E = eye(n);
else
   pivot = false;
end

% Initialize.

A = A.d;
minmn = min([m,n]);
R = zeros(minmn, n);
p = min([m-1,n]);
nrm = zeros(n,1);

% Triangularization loop.

for k=1:p

   if pivot

%     Pivoting requested. Compute the reduced column norms.

      for j=k:n
         nrm(j) = norm(A(k:m,j));
      end

%     Determine the pivot column jm and interchage columns k
%     and jm in A, Q, and E.

      [mx, jm] = max(nrm(k:n));
      jm = jm+k-1;
      temp = A(k:m,k);
      A(k:m,k) = A(k:m,jm);
      A(k:m,jm) = temp;
      temp = R(1:k-1,k);
      R(1:k-1,k) = R(1:k-1,jm);
      R(1:k-1,jm) = temp;
      temp = E(1:n,k);
      E(1:n,k) = E(1:n,jm);
      E(1:n,jm) = temp;
   end

%  Perform one step of the reduction, updating R.

   [u, rho] = FlapHousegen(A(k:m,k));
   A(k:m,k+1:n) = FlapHouseappl(u, A(k:m,k+1:n));
   R(k,k) = rho;
   R(k,k+1:n) = A(k,k+1:n);
   A(k:m,k) = u;
end

% If necessary update the last row of R.

if p==m-1
   R(m,m:n) = A(m,m:n);
end

% Compute Q by back multiplication of the Householder transformations.

if econ
   Q = eye(m, minmn);
else
   Q = eye(m);
   R = [R; zeros(m-n, n)];
end
nn = size(Q,2);

for k=p:-1:1
   Q(k:m,k:nn) = FlapHouseappl(A(k:m,k), Q(k:m,k:nn));
end

% Normalize Q and R.

for i=1:minmn
   srii = sign(R(i,i));
   if srii==0, srii=1; end
   Q(:,i) = FlapRound(Q(:,i)*srii);
   R(i,:) = FlapRound(conj(srii)*R(i,:));
   R(i,i) = real(R(i,i));
end

Q = flap(Q);
R = flap(R);
 


   