function h = FlapPlus(f, g)
%FlapPlus   implements + for double arrays taking cancellation into account.
%   FlapPlus implements f + g for doubles.  Because cancellation can promote
%   rounding error in f and g where it can be seen at FlapRlevel,
%   it is necessary to compute a smaller rounding level.  The process
%   is expensive, requiring and extra log of f+g.  When FlapRleve < 8,
%   this problem does not occur.

global FlapRlevel

if FlapRlevel<8 | FlapRlevel==16

   % Cancellation cannot cause problems in this range.
   % Compute the difference and round.

   
   h =  FlapRound(f + g);
   

else

   % Cancellation can cause rounding error from the basement
   % to rise above FlapRlevel.  When this occurs, round at a
   % lower level.

   hd = f + g;
   if isreal(hd)
      denom = max(abs(f),abs(g));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    1 + FlapRlevel + ceil(log10((abs(hd)+(hd==0))./denom)));
      h = FlapRound(hd, Rlevel);
   else
      hrd = real(hd);
      denom = max(abs(real(f)),  abs(real(g)));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    1 + FlapRlevel + ceil(log10((abs(hrd)+(hrd==0))./denom)));
      hrd = FlapRound(hrd, Rlevel);
      hid = imag(hd);
      denom = max(abs(imag(f)),  abs(imag(g)));
      denom = denom + (denom==0);
      Rlevel =  min(FlapRlevel, ...
                    1 + FlapRlevel + ceil(log10((abs(hid)+(hid==0))./denom)));
      hid = FlapRound(hid, Rlevel);
      h = complex(hrd, hid);
   end
end

