#lang racket
(provide test-runner test-runner-io)
(require rackunit)

(define (test-runner run)
  ;; Abscond examples
  (check-equal? (run 7) 7)
  (check-equal? (run -8) -8)

  ;; Blackmail examples
  (check-equal? (run '(add1 (add1 7))) 9)
  (check-equal? (run '(add1 (sub1 7))) 7)

  ;; Con examples
  (check-equal? (run '(if (zero? 0) 1 2)) 1)
  (check-equal? (run '(if (zero? 1) 1 2)) 2)
  (check-equal? (run '(if (zero? -7) 1 2)) 2)
  (check-equal? (run '(if (zero? 0)
                          (if (zero? 1) 1 2)
                          7))
                2)
  (check-equal? (run '(if (zero? (if (zero? 0) 1 0))
                          (if (zero? 1) 1 2)
                          7))
                7)

  ;; Dupe examples
  (check-equal? (run #t) #t)
  (check-equal? (run #f) #f)
  (check-equal? (run '(if #t 1 2)) 1)
  (check-equal? (run '(if #f 1 2)) 2)
  (check-equal? (run '(if 0 1 2)) 1)
  (check-equal? (run '(if #t 3 4)) 3)
  (check-equal? (run '(if #f 3 4)) 4)
  (check-equal? (run '(if  0 3 4)) 3)
  (check-equal? (run '(zero? 4)) #f)
  (check-equal? (run '(zero? 0)) #t)

  ;; Dodger examples
  (check-equal? (run #\a) #\a)
  (check-equal? (run #\b) #\b)
  (check-equal? (run '(char? #\a)) #t)
  (check-equal? (run '(char? #t)) #f)
  (check-equal? (run '(char? 8)) #f)
  (check-equal? (run '(char->integer #\a)) (char->integer #\a))
  (check-equal? (run '(integer->char 955)) #\λ)

  ;; Extort examples
  (check-equal? (run '(add1 #f)) 'err)
  (check-equal? (run '(sub1 #f)) 'err)
  (check-equal? (run '(zero? #f)) 'err)
  (check-equal? (run '(char->integer #f)) 'err)
  (check-equal? (run '(integer->char #f)) 'err)
  (check-equal? (run '(integer->char -1)) 'err)
  (check-equal? (run '(write-byte #f)) 'err)
  (check-equal? (run '(write-byte -1)) 'err)
  (check-equal? (run '(write-byte 256)) 'err)

  ;; Fraud examples
  (check-equal? (run '(let ((x 7)) x)) 7)
  (check-equal? (run '(let ((x 7)) 2)) 2)
  (check-equal? (run '(let ((x 7)) (add1 x))) 8)
  (check-equal? (run '(let ((x (add1 7))) x)) 8)
  (check-equal? (run '(let ((x 7)) (let ((y 2)) x))) 7)
  (check-equal? (run '(let ((x 7)) (let ((x 2)) x))) 2)
  (check-equal? (run '(let ((x 7)) (let ((x (add1 x))) x))) 8)

  (check-equal? (run '(let ((x 0))
                        (if (zero? x) 7 8)))
                7)
  (check-equal? (run '(let ((x 1))
                        (add1 (if (zero? x) 7 8))))
                9)
  (check-equal? (run '(+ 3 4)) 7)
  (check-equal? (run '(- 3 4)) -1)
  (check-equal? (run '(+ (+ 2 1) 4)) 7)
  (check-equal? (run '(+ (+ 2 1) (+ 2 2))) 7)
  (check-equal? (run '(let ((x (+ 1 2)))
                        (let ((z (- 4 x)))
                          (+ (+ x x) z))))
                7)
  (check-equal? (run '(= 5 5)) #t)
  (check-equal? (run '(= 4 5)) #f)
  (check-equal? (run '(= (add1 4) 5)) #t)
  (check-equal? (run '(< 5 5)) #f)
  (check-equal? (run '(< 4 5)) #t)
  (check-equal? (run '(< (add1 4) 5)) #f)

  ;; Hustle examples
  (check-equal? (run ''()) '())
  (check-equal? (run '(box 1)) (box 1))
  (check-equal? (run '(box -1)) (box -1))
  (check-equal? (run '(cons 1 2)) (cons 1 2))
  (check-equal? (run '(unbox (box 1))) 1)
  (check-equal? (run '(car (cons 1 2))) 1)
  (check-equal? (run '(cdr (cons 1 2))) 2)
  (check-equal? (run '(cons 1 '())) (list 1))
  (check-equal? (run '(let ((x (cons 1 2)))
                        (begin (cdr x)
                               (car x))))
                1)
  (check-equal? (run '(let ((x (cons 1 2)))
                        (let ((y (box 3)))
                          (unbox y))))
                3)
  (check-equal? (run '(eq? 1 1)) #t)
  (check-equal? (run '(eq? 1 2)) #f)
  (check-equal? (run '(eq? (cons 1 2) (cons 1 2))) #f)
  (check-equal? (run '(let ((x (cons 1 2))) (eq? x x))) #t)

  ;; Hoax examples
  (check-equal? (run '(make-vector 0 0)) #())
  (check-equal? (run '(make-vector 1 0)) #(0))
  (check-equal? (run '(make-vector 3 0)) #(0 0 0))
  (check-equal? (run '(make-vector 3 5)) #(5 5 5))
  (check-equal? (run '(vector? (make-vector 0 0))) #t)
  (check-equal? (run '(vector? (cons 0 0))) #f)
  (check-equal? (run '(vector-ref (make-vector 0 #f) 0)) 'err)
  (check-equal? (run '(vector-ref (make-vector 3 5) -1)) 'err)
  (check-equal? (run '(vector-ref (make-vector 3 5) 0)) 5)
  (check-equal? (run '(vector-ref (make-vector 3 5) 1)) 5)
  (check-equal? (run '(vector-ref (make-vector 3 5) 2)) 5)
  (check-equal? (run '(vector-ref (make-vector 3 5) 3)) 'err)
  (check-equal? (run '(let ((x (make-vector 3 5)))
                        (begin (vector-set! x 0 4)
                               x)))
                #(4 5 5))
  (check-equal? (run '(let ((x (make-vector 3 5)))
                        (begin (vector-set! x 1 4)
                               x)))
                #(5 4 5))
  (check-equal? (run '(vector-length (make-vector 3 #f))) 3)
  (check-equal? (run '(vector-length (make-vector 0 #f))) 0)
  (check-equal? (run '"") "")
  (check-equal? (run '"fred") "fred")
  (check-equal? (run '"wilma") "wilma")
  (check-equal? (run '(make-string 0 #\f)) "")
  (check-equal? (run '(make-string 3 #\f)) "fff")
  (check-equal? (run '(make-string 3 #\g)) "ggg")
  (check-equal? (run '(string-length "")) 0)
  (check-equal? (run '(string-length "fred")) 4)
  (check-equal? (run '(string-ref "" 0)) 'err)
  (check-equal? (run '(string-ref (make-string 0 #\a) 0)) 'err)
  (check-equal? (run '(string-ref "fred" 0)) #\f)
  (check-equal? (run '(string-ref "fred" 1)) #\r)
  (check-equal? (run '(string-ref "fred" 2)) #\e)
  (check-equal? (run '(string-ref "fred" 4)) 'err)
  (check-equal? (run '(string? "fred")) #t)
  (check-equal? (run '(string? (cons 1 2))) #f)
  (check-equal? (run '(begin (make-string 3 #\f)
                             (make-string 3 #\f)))
                "fff")

  ;; Iniquity tests
  (check-equal? (run
                 '(define (f x) x)
                 '(f 5))
                5)
  (check-equal? (run
                 '(define (tri x)
                    (if (zero? x)
                        0
                        (+ x (tri (sub1 x)))))
                 '(tri 9))
                45)

  (check-equal? (run
                 '(define (f x) x)
                 '(define (g x) (f x))
                 '(g 5))
                5)
  (check-equal? (run
                 '(define (my-even? x)
                    (if (zero? x)
                        #t
                        (my-odd? (sub1 x))))
                 '(define (my-odd? x)
                    (if (zero? x)
                        #f
                        (my-even? (sub1 x))))
                 '(my-even? 101))
                #f)
  (check-equal? (run
                 '(define (map-add1 xs)
                    (if (empty? xs)
                        '()
                        (cons (add1 (car xs))
                              (map-add1 (cdr xs)))))
                 '(map-add1 (cons 1 (cons 2 (cons 3 '())))))
                '(2 3 4))
  (check-equal? (run
                 '(define (f x)
                    10)
                 '(f 1))
                10)
  (check-equal? (run
                 '(define (f x)
                     10)
                 '(let ((x 2)) (f 1)))
                10)
  (check-equal? (run
                 '(define (f x y)
                    10)
                 '(f 1 2))
                10)
  (check-equal? (run
                 '(define (f x y)
                    10)
                 '(let ((z 2)) (f 1 2)))
                10)

  ;; Knock examples
  (check-equal? (run '(match 1)) 'err)
  (check-equal? (run '(match 1 [1 2]))
                2)
  (check-equal? (run '(match 1 [2 1] [1 2]))
                2)
  (check-equal? (run '(match 1 [2 1] [1 2] [0 3]))
                2)
  (check-equal? (run '(match 1 [2 1] [0 3]))
                'err)
  (check-equal? (run '(match 1 [_ 2] [_ 3]))
                2)
  (check-equal? (run '(match 1 [x 2] [_ 3]))
                2)
  (check-equal? (run '(match 1 [x x] [_ 3]))
                1)
  (check-equal? (run '(match (cons 1 2) [x x] [_ 3]))
                (cons 1 2))
  (check-equal? (run '(match (cons 1 2) [(cons x y) x] [_ 3]))
                1)
  (check-equal? (run '(match (cons 1 2) [(cons x 2) x] [_ 3]))
                1)
  (check-equal? (run '(match (cons 1 2) [(cons 3 2) 0] [_ 3]))
                3)
  (check-equal? (run '(match 1 [(cons x y) x] [_ 3]))
                3)
  (check-equal? (run '(match (cons 1 2) [(cons 1 3) 0] [(cons 1 y) y] [_ 3]))
                2)
  (check-equal? (run '(match (box 1) [(box 1) 0] [_ 1]))
                0)
  (check-equal? (run '(match (box 1) [(box 2) 0] [_ 1]))
                1)
  (check-equal? (run '(match (box 1) [(box x) x] [_ 2]))
                1)

  ;; Loot examples
  (check-true (procedure? (run '(λ (x) x))))
  (check-equal? (run '((λ (x) x) 5))
                5)

  (check-equal? (run '(let ((f (λ (x) x))) (f 5)))
                5)
  (check-equal? (run '(let ((f (λ (x y) x))) (f 5 7)))
                5)
  (check-equal? (run '(let ((f (λ (x y) y))) (f 5 7)))
                7)
  (check-equal? (run '((let ((x 1))
                         (let ((y 2))
                           (lambda (z) (cons x (cons y (cons z '()))))))
                       3))
                '(1 2 3))
  (check-equal? (run '(define (adder n)
                        (λ (x) (+ x n)))
                     '((adder 5) 10))
                15)
  (check-equal? (run '(((λ (t)
                          ((λ (f) (t (λ (z) ((f f) z))))
                           (λ (f) (t (λ (z) ((f f) z))))))
                        (λ (tri)
                          (λ (n)
                            (if (zero? n)
                                0
                                (+ n (tri (sub1 n)))))))
                       36))
                666)
  (check-equal? (run '(define (tri n)
                        (if (zero? n)
                            0
                            (+ n (tri (sub1 n)))))
                     '(tri 36))
                666)
  (check-equal? (run '(define (tri n)
                        (match n
                          [0 0]
                          [m (+ m (tri (sub1 m)))]))
                     '(tri 36))
                666)
  (check-equal? (run '((match 8 [8 (lambda (x) x)]) 12))
                12)

  ;; Mug examples
  (check-equal? (run '(symbol? 'foo)) #t)
  (check-equal? (run '(symbol? (string->symbol "foo"))) #t)
  (check-equal? (run '(eq? 'foo 'foo)) #t)
  (check-equal? (run '(eq? (string->symbol "foo")
                           (string->symbol "foo")))
                #t)
  (check-equal? (run '(eq? 'foo (string->symbol "foo")))
                #t)
  (check-equal? (run '(eq? 'fff (string->symbol (make-string 3 #\f))))
                #t)
  (check-equal? (run '(symbol? 'g0)) #t)
  (check-equal? (run '(symbol? "g0")) #f)
  (check-equal? (run '(symbol? (string->symbol "g0"))) #t)
  (check-equal? (run '(symbol? (string->uninterned-symbol "g0"))) #t)
  (check-equal? (run '(eq? 'g0 (string->symbol "g0"))) #t)
  (check-equal? (run '(eq? 'g0 (string->uninterned-symbol "g0"))) #f)
  (check-equal? (run '(eq? (string->uninterned-symbol "g0") (string->uninterned-symbol "g0")))
                #f)
  (check-equal? (run '(eq? (symbol->string 'foo) (symbol->string 'foo))) #f)
  (check-equal? (run '(string? (symbol->string 'foo))) #t)
  (check-equal? (run '(eq? (symbol->string 'foo) "foo")) #f)
  (check-equal? (run ''foo) 'foo)
  (check-equal? (run '(eq? (match #t [_ "foo"]) "bar")) #f)
  (check-equal? (run '(eq? (match #t [_ 'foo]) 'bar)) #f)
  (check-equal? (run '(match 'foo ['bar #t] [_ #f])) #f)
  (check-equal? (run '(match 'foo ['foo #t] [_ #f])) #t)
  (check-equal? (run '(match "foo" ["foo" #t] [_ #f])) #t)
  (check-equal? (run '(match "foo" ["bar" #t] [_ #f])) #f)
  (check-equal? (run '(match (cons '+ (cons 1 (cons 2 '())))
                        [(cons '+ (cons x (cons y '())))
                         (+ x y)]))
                3)

  ;; Mountebank examples
  (check-equal? (run '#())
                #())
  (check-equal? (run ''#())
                #())
  (check-equal? (run ''#t)
                #t)
  (check-equal? (run ''7)
                7)
  (check-equal? (run ''(1 2 3))
                '(1 2 3))
  (check-equal? (run ''(1 . 2))
                '(1 . 2))
  (check-equal? (run ''(("1") (#() #(1 #(2))) (#&(1)) (#f) (4) (5)))
                '(("1") (#() #(1 #(2))) (#&(1)) (#f) (4) (5)))
  (check-equal? (run '(define (f) (cons 1 2))
                     '(eq? (f) (f)))
                #f)
  (check-equal? (run '(define (f) '(1 . 2))
                     '(eq? (f) (f)))
                #t)
  (check-equal? (run '(let ((x '(foo . foo)))
                        (eq? (car x) (cdr x))))
                #t)
  (check-equal?
   (run '(define (eval e r)
           (match e
             [(list 'zero? e)
              (zero? (eval e r))]
             [(list 'sub1 e)
              (sub1 (eval e r))]
             [(list '+ e1 e2)
              (+ (eval e1 r) (eval e2 r))]
             [(list 'if e1 e2 e3)
              (if (eval e1 r)
                  (eval e2 r)
                  (eval e3 r))]
             [(list 'λ (list x) e)
              (lambda (v) (eval e (cons (cons x v) r)))]
             [(list e1 e2)
              ((eval e1 r) (eval e2 r))]
             [_
              (if (symbol? e)
                  (lookup r e)
                  e)]))
        '(define (lookup r x)
           (match r
             [(cons (cons y v) r)
              (if (eq? x y)
                  v
                  (lookup r x))]))
        '(eval '(((λ (t)
                    ((λ (f) (t (λ (z) ((f f) z))))
                     (λ (f) (t (λ (z) ((f f) z))))))
                  (λ (tri)
                    (λ (n)
                      (if (zero? n)
                          0
                          (+ n (tri (sub1 n)))))))
                 36)
               '()))
   666)

  ;; Neerdowell examples
  (check-equal? (run '(struct foo ())
                     '(foo? (foo)))
                #t)
  (check-equal? (run '(struct foo (x))
                     '(foo? (foo 1)))
                #t)
  (check-equal? (run '(struct foo ())
                     '(struct bar ())
                     '(foo? (bar)))
                #f)
  (check-equal? (run '(struct foo ())
                     '(struct bar ())
                     '(bar? (bar)))
                #t)
  (check-equal? (run '(struct foo ())
                     '(struct bar ())
                     '(bar? #()))
                #f)
  (check-equal? (run '(struct foo (x))
                     '(foo-x (foo 3)))
                3)
  (check-equal? (run '(struct foo (x))
                     '(let ((x (foo 3)))
                        (foo-x x)))
                3)
  (check-equal? (run '(struct foo (x))
                     '(let ((x (foo 3)))
                        (foo-x x)))
                3)
  (check-equal? (run '(struct foo (x))
                     '(let ((x (foo (foo 3))))
                        (foo? (foo-x x))))
                #t)
  (check-equal? (run '(struct foo (x y z))
                     '(let ((x (foo 1 2 3)))
                        (cons (foo-x x)
                              (cons (foo-y x)
                                    (cons (foo-z x)
                                          '())))))
                '(1 2 3))
  (check-equal? (run '(struct foo ())
                     '(eq? (foo) (foo)))
                #f)
  (check-equal? (run '(struct foo (x))
                     '(foo-x #t))
                'err)
  (check-equal? (run '(struct foo (x))
                     '(struct bar (y))
                     '(match (bar 5)
                        [(foo x) #f]
                        [(bar x) x]))
                5)
  (check-equal? (run '(struct nil ())
                     '(struct pair (x y))
                     '(define (len x)
                        (match x
                          [(nil) 0]
                          [(pair _ x) (add1 (len x))]))
                     '(len (pair 1 (pair 2 (pair 3 (nil))))))
                3)
  (check-equal? (run '(match (cons (cons 1 2) '())
                        [(cons (cons x y) '()) y]))
                2)
  (check-equal? (run '(struct foo (p q))
                     '(match (cons (foo 1 2) '())
                        [(cons (foo x y) _) y]))
                2)
  (check-equal? (run '(struct foo (p q))
                     '(match (cons (foo 1 2) '())
                        [(cons (foo x 3) _) x]
                        [_ 9]))
                9)
  (check-equal? (run '(struct foo (x q))
                     '(define (get z)
                        (match z
                          ['() #f]
                          [(cons (foo x q) y) x]))
                     '(get (cons (foo 7 2) '())))
                7)
  (check-equal? (run '(struct posn (x y))
                     '(define (posn-xs ps)
                        (match ps
                          ['() '()]
                          [(cons (posn x y) ps)
                           (cons x (posn-xs ps))]))
                     '(posn-xs (cons (posn 3 4) (cons (posn 5 6) (cons (posn 7 8) '())))))
                '(3 5 7))
  (check-equal? (run '(struct Foo (x y z))
                     '(match (Foo 1 2 3)
                        [(Foo x y z) z]))
                3)
  (check-equal? (run '(struct Boo (x))
                     '(match 8
                        [(Boo 'y) 0]
                        [_ 1]))
                1)

  ;; Outlaw examples
  (check-equal? (run '(+)) 0)
  (check-equal? (run '(+ 1 2 3)) 6)
  (check-equal? (run '(< 1)) #t)
  (check-equal? (run '(< 1 2 3)) #t)
  (check-equal? (run '(< 1 3 3)) #f)
  (check-equal? (run '(> 1)) #t)
  (check-equal? (run '(> 3 2 1)) #t)
  (check-equal? (run '(> 3 3 1)) #f)
  (check-equal? (run '(<= 1)) #t)
  (check-equal? (run '(<= 1 2 3)) #t)
  (check-equal? (run '(<= 1 3 3)) #t)
  (check-equal? (run '(<= 1 4 3)) #f)
  (check-equal? (run '(>= 1)) #t)
  (check-equal? (run '(>= 3 2 1)) #t)
  (check-equal? (run '(>= 3 3 1)) #t)
  (check-equal? (run '(>= 3 4 1)) #f)
  (check-equal? (run '(list)) '())
  (check-equal? (run '(list 1 2 3)) '(1 2 3))
  (check-equal? (run '(map add1 (list 1 2 3))) '(2 3 4))
  (check-equal? (run '(map + (list 1 2 3) (list 4 5 6))) '(5 7 9))
  (check-equal? (run '(append)) '())
  (check-equal? (run '(append '(1 2 3))) '(1 2 3))
  (check-equal? (run '(append '(1 2 3) '())) '(1 2 3))
  (check-equal? (run '(append '() '(1 2 3))) '(1 2 3))
  (check-equal? (run '(append '(1 2 3) '(4 5 6))) '(1 2 3 4 5 6))
  (check-equal? (run '(memq 'x '())) #f)
  (check-equal? (run '(memq 'x '(p x y))) '(x y))
  (check-equal? (run '(member 'x '() eq?)) #f)
  (check-equal? (run '(member 'x '(p x y) eq?)) '(x y))
  (check-equal? (run '(append-map list '(1 2 3))) '(1 2 3))
  (check-equal? (run '(vector->list #())) '())
  (check-equal? (run '(vector->list #(1 2 3))) '(1 2 3))
  (check-equal? (run '(number->string 0)) "0")
  (check-equal? (run '(number->string 10)) "10")
  (check-equal? (run '(number->string 123)) "123")
  (check-equal? (run '(number->string 0 10)) "0")
  (check-equal? (run '(number->string 10 10)) "10")
  (check-equal? (run '(number->string 123 10)) "123")
  (check-equal? (run '(number->string 0 2)) "0")
  (check-equal? (run '(number->string 1 2)) "1")
  (check-equal? (run '(number->string 3 2)) "11")
  (check-equal? (run '(number->string 8 2)) "1000")
  (check-equal? (run '(number->string 0 8)) "0")
  (check-equal? (run '(number->string 1 8)) "1")
  (check-equal? (run '(number->string 3 8)) "3")
  (check-equal? (run '(number->string 8 8)) "10")
  (check-equal? (run '(number->string 0 16)) "0")
  (check-equal? (run '(number->string 1 16)) "1")
  (check-equal? (run '(number->string 3 16)) "3")
  (check-equal? (run '(number->string 8 16)) "8")
  (check-equal? (run '(number->string 10 16)) "a")
  (check-equal? (run '(number->string 15 16)) "f")
  (check-equal? (run '(number->string 16 16)) "10")
  (check-pred symbol? (run '(gensym)))
  (check-equal? (run '(eq? (gensym) (gensym))) #f)
  (check-equal? (run '(let ((x (gensym))) (eq? x x))) #t)
  (check-pred symbol? (run '(gensym 'fred)))
  (check-equal? (run '(eq? (gensym 'fred) (gensym 'fred))) #f)
  (check-equal? (run '(let ((x (gensym 'fred))) (eq? x x))) #t)
  (check-pred symbol? (run '(gensym "fred")))
  (check-equal? (run '(eq? (gensym "fred") (gensym "fred"))) #f)
  (check-equal? (run '(let ((x (gensym "fred"))) (eq? x x))) #t)
  (check-equal? (run '(void? (void))) #t)
  (check-equal? (run '(void? void)) #f)
  (check-equal? (run '(eq? (void) (void))) #t)
  (check-equal? (run '(bitwise-and #b111 #b000)) #b000)
  (check-equal? (run '(bitwise-and #b111 #b111)) #b111)
  (check-equal? (run '(bitwise-and #b101 #b100)) #b100)
  (check-equal? (run '(bitwise-and #b001 #b100)) #b000)
  (check-equal? (run '(bitwise-ior #b111 #b000)) #b111)
  (check-equal? (run '(bitwise-ior #b111 #b111)) #b111)
  (check-equal? (run '(bitwise-ior #b101 #b100)) #b101)
  (check-equal? (run '(bitwise-ior #b001 #b100)) #b101)
  (check-equal? (run '(bitwise-xor #b111 #b000)) #b111)
  (check-equal? (run '(bitwise-xor #b111 #b111)) #b000)
  (check-equal? (run '(bitwise-xor #b101 #b100)) #b001)
  (check-equal? (run '(bitwise-xor #b001 #b100)) #b101)
  (check-equal? (run '(arithmetic-shift 1 0)) 1)
  (check-equal? (run '(arithmetic-shift 1 1)) 2)
  (check-equal? (run '(arithmetic-shift 1 2)) 4)
  (check-equal? (run '(arithmetic-shift 1 3)) 8)
  (check-equal? (run '(arithmetic-shift 3 2)) 12)
  (check-equal? (run '(or)) #f)
  (check-equal? (run '(or #t)) #t)
  (check-equal? (run '(or 7)) 7)
  (check-equal? (run '(or 7 #t)) 7)
  (check-equal? (run '(or #f #f #f)) #f)
  (check-equal? (run '(or #f 7 9)) 7)
  (check-equal? (run '(list->string '())) "")
  (check-equal? (run '(list->string '(#\a #\b #\c))) "abc")
  (check-equal? (run '(char<=? #\a)) #t)
  (check-equal? (run '(char<=? #\a #\b)) #t)
  (check-equal? (run '(char<=? #\a #\b #\c)) #t)
  (check-equal? (run '(char<=? #\a #\b #\b)) #t)
  (check-equal? (run '(char<=? #\a #\b #\a)) #f)
  (check-equal? (run '(= (eq-hash-code 'x) (eq-hash-code 'x))) #t)
  (check-equal? (run '(= (eq-hash-code 'x) (eq-hash-code 'y))) #f)
  (check-equal? (run '(foldr + #f '())) #f)
  (check-equal? (run '(foldr + 0 '(1 2 3))) 6)
  (check-equal? (run '(list? '())) #t)
  (check-equal? (run '(list? '(1 2 3))) #t)
  (check-equal? (run '(list? (cons 1 2))) #f)
  (check-equal? (run '(list? #t)) #f)
  (check-equal? (run '(reverse '())) '())
  (check-equal? (run '(reverse '(1 2 3))) '(3 2 1))
  (check-equal? (run '(remove-duplicates '() eq?)) '())
  (check-equal? (run '(remove-duplicates '(1 2 3) eq?)) '(1 2 3))
  (check-equal? (run '(remove-duplicates '(1 2 3 2 1 3) eq?)) '(1 2 3))
  (check-equal? (run '(remove 'x '() eq?)) '())
  (check-equal? (run '(remove 'x '(x y z) eq?)) '(y z))
  (check-equal? (run '(remove 'x '(p q x r) eq?)) '(p q r))
  (check-equal? (run '(remove 'x '(p q x r x) eq?)) '(p q r x))
  (check-equal? (run '(remove* 'x '() eq?)) '())
  (check-equal? (run '(remove* 'x '(x y z) eq?)) '(y z))
  (check-equal? (run '(remove* 'x '(p q x r) eq?)) '(p q r))
  (check-equal? (run '(remove* 'x '(p q x r x) eq?)) '(p q r))
  (check-equal? (run '(remq* '(x y) '())) '())
  (check-equal? (run '(remq* '(x y) '(x y z))) '(z))
  (check-equal? (run '(remq* '(x y) '(p q x r x))) '(p q r))
  (check-equal? (run '(make-list 0 #\a)) '())
  (check-equal? (run '(make-list 3 #\a)) '(#\a #\a #\a))
  (check-equal? (run '(match 8
                        [(? integer?) 1]
                        [_ 2]))
                1)
  (check-equal? (run '(match 8
                        [(? string?) 1]
                        [_ 2]))
                2)
  (check-equal? (run '(match (cons 8 "8")
                        [(cons (? integer?) (? string?)) 1]
                        [_ 2]))
                1)
  (check-equal? (run '(match 8
                        [(? (lambda (x) (eq? x 8))) 1]
                        [_ 2]))
                1)
  (check-equal? (run '(match 8
                        [(? integer? x) x]
                        [_ 2]))
                8)
  (check-equal? (run '(match (box #\a)
                        [(box (and x (? integer?))) 1]
                        [(box (and x (? char?))) x]))
                #\a)

  (check-equal? (run '(vector)) #())
  (check-equal? (run '(vector 1 2 3)) #(1 2 3))
  (check-equal? (run '(list->vector '())) #())
  (check-equal? (run '(list->vector '(1 2 3))) #(1 2 3))
  (check-equal? (run '(boolean? #t)) #t)
  (check-equal? (run '(boolean? #f)) #t)
  (check-equal? (run '(boolean? 8)) #f)
  (check-equal? (run '(substring "hello" 0)) "hello")
  (check-equal? (run '(substring "hello" 1)) "ello")
  (check-equal? (run '(substring "hello" 1 4)) "ell")
  (check-equal? (run '(odd? 7)) #t)
  (check-equal? (run '(odd? 8)) #f)
  (check-equal? (run '(filter odd? '())) '())
  (check-equal? (run '(filter odd? '(1 2 3 4))) '(1 3))
  (check-equal? (run '(findf odd? '())) #f)
  (check-equal? (run '(findf odd? '(2 4 3 7))) 3)
  (check-equal? (run '(char-alphabetic? #\a)) #t)
  (check-equal? (run '(char-alphabetic? #\space)) #f)
  (check-equal? (run '(char-whitespace? #\a)) #f)
  (check-equal? (run '(char-whitespace? #\space)) #t)
  (check-equal? (run '(begin 1)) 1)
  (check-equal? (run '(begin 1 2)) 2)
  (check-equal? (run '(begin 1 2 3)) 3)
  (check-equal? (run '(let () 1 2)) 2)
  (check-equal? (run '(let ((x 1)) x x)) 1)
  (check-equal? (run '(let ((x 1)) x x x)) 1)
  (check-equal? (run '(match 1 [1 2 3])) 3)
  (check-equal? (run '(system-type)) (system-type))
  (check-equal? (run '(struct Foo (x))
                     '(struct Bar (y))
                     '(match (Bar 1)
                        [(Foo x) #f]
                        [(Bar x) x]))
                1)
  (check-equal? (run '(procedure? add1)) #t)
  (check-equal? (run '(procedure? (lambda (x) x))) #t)
  (check-equal? (run '(procedure? 8)) #f)
  (check-equal? (run '(struct posn (x y))
                     '(procedure? (posn 3 4)))
                #f)
  (check-equal? (run '(apply string-append (list "x")))
                "x")

  (check-equal? (run '(* 0 8)) 0)
  (check-equal? (run '(* 1 8)) 8)
  (check-equal? (run '(* 2 9)) 18)
  (check-equal? (run '(* 2 -3)) -6)
  (check-equal? (run '(* 4 3)) 12)
  (check-equal? (run '(* 8 3)) 24)
  (check-equal? (run '(* 16 2)) 32)
  (check-equal? (run '(* 10 5)) 50)
  (check-equal? (run '(* 64 2)) 128)
  (check-equal? (run '(let ((pred (lambda (x) #t)))
                        (match 0
                          [(and (? pred) _) #t]
                          [_ #f])))
                #t))


(define (test-runner-io run)
  ;; Evildoer examples
  (check-equal? (run "" 7) (cons 7 ""))
  (check-equal? (run "" '(write-byte 97)) (cons (void) "a"))
  (check-equal? (run "a" '(read-byte)) (cons 97 ""))
  (check-equal? (run "b" '(begin (write-byte 97) (read-byte)))
                (cons 98 "a"))
  (check-equal? (run "" '(read-byte)) (cons eof ""))
  (check-equal? (run "" '(eof-object? (read-byte))) (cons #t ""))
  (check-equal? (run "a" '(eof-object? (read-byte))) (cons #f ""))
  (check-equal? (run "" '(begin (write-byte 97) (write-byte 98)))
                (cons (void) "ab"))
  (check-equal? (run "ab" '(peek-byte)) (cons 97 ""))
  (check-equal? (run "ab" '(begin (peek-byte) (read-byte))) (cons 97 ""))
  ;; Extort examples
  (check-equal? (run "" '(write-byte #t)) (cons 'err ""))

  ;; Fraud examples
  (check-equal? (run "" '(let ((x 97)) (write-byte x))) (cons (void) "a"))
  (check-equal? (run ""
                     '(let ((x 97))
                        (begin (write-byte x)
                               x)))
                (cons 97 "a"))
  (check-equal? (run "b" '(let ((x 97)) (begin (read-byte) x)))
                (cons 97 ""))
  (check-equal? (run "b" '(let ((x 97)) (begin (peek-byte) x)))
                (cons 97 ""))

  ;; Hustle examples
  (check-equal? (run ""
                     '(let ((x 1))
                        (begin (write-byte 97)
                               1)))
                (cons 1 "a"))

  (check-equal? (run ""
                     '(let ((x 1))
                        (let ((y 2))
                          (begin (write-byte 97)
                                 1))))
                (cons 1 "a"))

  (check-equal? (run ""
                     '(let ((x (cons 1 2)))
                        (begin (write-byte 97)
                               (car x))))
                (cons 1 "a"))
  ;; Iniquity examples
  (check-equal? (run ""
                     '(define (print-alphabet i)
                        (if (zero? i)
                            (void)
                            (begin (write-byte (- 123 i))
                                   (print-alphabet (sub1 i)))))
                     '(print-alphabet 26))
                (cons (void) "abcdefghijklmnopqrstuvwxyz"))

  ;; Outlaw examples
  (check-equal? (run "" '(read-char))
                (cons eof ""))
  (check-equal? (run "a" '(read-char))
                (cons #\a ""))
  (check-equal? (run "ab" '(read-char))
                (cons #\a ""))
  (check-equal? (run "ab" '(cons (read-char) (read-char)))
                (cons '(#\a . #\b) ""))
  (check-equal? (run "a" '(peek-byte (%current-input-port) 0))
                (cons 97 ""))
  (check-equal? (run "ab" '(cons (peek-byte (%current-input-port) 1) (read-byte)))
                (cons (cons 98 97) ""))
  (check-equal? (run "abc" '(cons (peek-byte (%current-input-port) 2)
                                  (cons (read-byte) (read-byte))))
                (cons (cons 99 (cons 97 98)) ""))
  (check-equal? (run "a" '(peek-char))
                (cons #\a ""))
  (check-equal? (run "ab" '(cons (peek-char) (peek-char)))
                (cons '(#\a . #\a) ""))
  (check-equal? (run "λ" '(peek-char))
                (cons #\λ ""))
  (check-equal? (run "" '(write-char #\a))
                (cons (void) "a"))
  (check-equal? (run "" '(write-char #\newline))
                (cons (void) "\n"))
  (check-equal? (run "" '(write-string "hello world"))
                (cons 11 "hello world"))
  (check-equal? (run "" '(displayln "hello world"))
                (cons (void) "hello world\n"))
  )
