package example1;

/* 
 * The Tetris class implements the Game interface
 * therefore it must implement all the methods the
 * interface defines.  It can also have additional
 * methods and of course, can define its own instance
 * variables.
 */
public class Tetris implements Game {
	/*
	 * We can initialize variables at declaration time, but it is best to initialize
	 * them in the constructor
	 */
	private int rotationNumber, position = 1, popularity, version;

	public Tetris(int version) {
		/*
		 * We need this. to tell the instance variable apart from the parameter.
		 */
		this.version = version;
		/*
		 * It is not necessary to set to 0, but we do it so programmers reading the code
		 * see we really mean for it to be 0. This is a good choice!
		 */
		rotationNumber = 0;

		/*
		 * We could have used this.popularity, but that is not necessary as there is no
		 * other popularity variable (e.g., a parameter) in the method.
		 */
		popularity = 0;

		/*
		 * We should have initialized position here instead of declaration time.
		 */
	}

	/* 
	 * Does this class has a default constructor?.  No! By providing a 
	 * constructor (no matter which one) no default constructor will be 
	 * provided by Java.  The default constructor provided by Java does 
	 * nothing.  If you provide a constructor other than the default 
	 * constructor, and want a default constructor, you must explicitly 
	 * define it.
	 */
	
	/*
	 * This method returns a reference to the current object. The reference "this"
	 * represents the object dropOnce() is operating on. Returning the current
	 * object allows cascading dropOnce() calls (e.g., t.dropOnce().dropOnce() where
	 * t is a Tetris object.
	 */
	public Tetris dropOnce() {
		position++;

		return this;
	}

	public Tetris rotateOnce() {
		rotationNumber++;

		return this;
	}

	/*
	 * Must be implemented as are defined by Game interface. /* Our class will not
	 * compile otherwise.
	 */
	public void play() {
		dropOnce().rotateOnce();
	}

	public int getScore() {
		/* Made up score as we need an integer :) */

		return position * rotationNumber;
	}

	public void setPopularity(int value) {
		popularity = value;
	}

	public int getPopularity() {
		return popularity;
	}

	/*
	 * What is the @Override below?. It is an annotation. When you override, you
	 * redefine what an existing method does. Every class has a toString() method
	 * that that resides in class called Object. The Object class is extended by all
	 * classes implicitly. The toString() method usually prints the class name and
	 * the address. That is not good enough for us, therefore we need to change what
	 * it does. When you provide a toString() method your definition will hide the
	 * original definition. The @Override makes sure you are overriding an existing
	 * method. Watch how the toString() method stops compiling when you add a
	 * parameter; why? Because by adding a parameter you are defining a new method,
	 * that does NOT overrides an existing one (the existing one has no parameters).
	 * When you override you must define a method with the same name and parameters
	 * as the original. The method that you override must exist in another class.
	 */
	@Override
	public String toString() {
		return "Tetris [rotationNumber=" + rotationNumber + ", position=" + position + ", popularity=" + popularity
				+ ", version=" + version + "]";
	}

	/*
	 * equals is another example of overriding a method that is part of the Object
	 * class. The equals method in the Object compares the address of two objects,
	 * not the contents. For example, if we have two Tetris objects references, t1
	 * and t2, if we don't provide an equals method, t1.equals(t2) is equivalent to
	 * t1 == t2. If you want to compare the contents of objects, you need to
	 * overwrite the equals method. Watch what happens when you change the parameter
	 * from Object obj to Tetris obj.
	 */
	@Override
	public boolean equals(Object obj) {
		/*
		 * When you implement an equals method the next four lines are 
		 * always the same (someone said free points?").
		 */

		if (obj == null) // If the parameter is null, the current object cannot
			return false; // equal to null, therefore is false
		if (obj.getClass() != getClass())
			return false;
		
		/* 
		 * About getClass() method: this method is part of the Object class
		 * and every class in Java can call it.  What it returns? It returns
		 * the address of an object that provides information about a class.
		 * When you define a class, an object is created that provides information
		 * about the class (there is a single object per class).  If you want
		 * to know whether two objects belong to the same class, you can just
		 * compare the addresses returned by getClass(). If the addresses are
		 * the same, then both objects share the same object that describes the
		 * class, what implies that they belong to the same class. The
		 * obj.getClass() != getClass() above says: do these two objects belong
		 * to the same class?  If not, then they cannot be equal.
		 * 
		 * Extra information: what is the class associated with the object
		 * that describes a class?  The name of that class is "Class".  
		 */
		
		Tetris tetris = (Tetris) obj;

		/*
		 * Two tetris objects will be considered the same if they have the same version
		 * number. Someone has to tell you what does it mean for two objects to be equal
		 * so you can implement the expected code.
		 */
		return version == tetris.version;
	}
}
