/* tssnetlib_0.h */

/* Network creation functions __________________________________________*/

/* Return a pointer to an empty network. */
void* create_network();

/* Delete network *ntwrk. */
void delete_network(void *ntwrk);

/* Add a node (with id) nid to network *ntwrk.
   Return 0 on success.
   Return -1 if node (with id) nid already exists, and print explanation.
   Return -2 if memory allocation failed, and print explanation. */
int add_node(void *ntwrk, int nid);

/* Add a one-way link to network *ntwrk:
   - srcnid: source node id; node must exist.
   - dstnid: destination node id; node must exist, differ from srcnid,
     a srcnid-to-dstnid link must not already exist.
   - rate (bytes/sec): transmit rate; must be > 0.
   - cvsq: coefficient of variation squared of rate; must be > 0;
     reasonable value is 0.05.
   - buffersize (bytes): number of bytes that buffer can hold; must be > 0.
   - propdelay (sec): propagation delay; must be >= 0.02
   - cost: cost of link; used for default (shortest-path) routing;
           must be > 0.
   Return id (> 0) of the link if success.
   Return -1 if a parameter is invalid and print explanation.
   Return -2 if memory allocation failed and print explanation. */
int add_link(void *ntwrk, int srcnid, int dstnid,
             double rate, double cvsq, int buffersize,
	     double propdelay, int cost);

/* Set the default (shortest-path) routes in network *ntwrk. */
void set_default_routes(void *ntwrk);

/* Print the default route from nodes srcnid to dstnid in network *ntwrk.
   Return 0 on success.
   Return -1 if a parameter is invalid and print explanation. */
int print_default_route(void *ntwrk, int srcnid, int dstnid);

/* Print the (default) routing table of node nid in network *ntwrk.
   Return 0 if success.
   Return -1 if a parameter is invalid, and print explanation. */
int print_routingtable(void *ntwrk, int nid);

/* Add a tcp flow using default routes to network *ntwrk:
   - srctype: type of tcp src; 1 for NewReno, 2 for Reno, 3 for Tahoe.
   - srcnid: id of the node where tcpsrc is located.
   - dstnid: id of the node where tcpsnk is located.
   - starttime (sec): time when the tcpsrc starts sending data.
   - datasize (bytes): amount of data that tcpsrc transfers to tcpsnk.
   Return id (> 0) of the tcp flow if success;
     flow's src has same id; id of flow's snk is id+1.
   Return -1 if a parameter is invalid, and print explanation.
   Return -2 if memory allocation failed, and print explanation. */
int add_tcp(void *ntwrk, int srctype, int srcnid, int snknid,
	    double starttime, int datasize);

/* Add a tcp flow using specified datapath and ackpath to network *ntwrk:
   - srctype: type of tcp src; 1 for NewReno, 2 for Reno, 3 for Tahoe.
   - datapath: array of ids of nodes on the data path (includes end nodes).
   - datapathlen: length of datapath
   - ackpath: array of ids of nodes on the ack path (includes end nodes).
   - ackpathlen: length of ackpath
   - starttime (sec): time when the tcpsrc starts sending data.
   - datasize (bytes): amount of data that tcpsrc transfers to tcpsnk.
   Return id (> 0) of the tcp flow;
     flow's src has same id; id of flow's snk is id+1.
   Return -1 if a parameter is invalid, and print explanation.
   Return -2 if memory allocation failed, and print explanation. */
int add_tcp_with_route(void *ntwrk, int srctype,
		       int datapath[], int datapathlen,
		       int ackpath[], int ackpathlen,
		       double starttime, int datasize);

/* Add a udp flow using default route to network *ntwrk:
   - srcnid: id of the node where udpsrc is located.
   - dstnid: id of the node where udpsnk is located.
   - srcrate (bytes/sec): rate at which udpsrc sends data.
   - starttime (sec): time when the udpsrc starts sending data.
   - datasize (bytes): amount of data that udpsrc sends.
   Return id (> 0) of the tcp flow;
     flow's src has same id; id of flow's snk is id+1.
   Return -1 if a parameter is invalid, and print explanation.
   Return -2 if memory allocation failed, and print explanation. */
int add_udp(void *ntwrk, int srcnid, int snknid,
	    double srcrate, double starttime, int datasize);

/* Add a udp flow using specified route to network *ntwrk:
   - datapath: array of ids of nodes on the data path (includes end nodes).
   - datapathlen: length of datapath
   - srcrate (bytes/sec): rate at which udpsrc sends data.
   - starttime (sec): time when the udpsrc starts sending data.
   - datasize (bytes): amount of data that udpsrc sends.
   Return id (> 0) of the tcp flow;
     flow's src has same id; id of flow's snk is id+1.
   Return -1 if a parameter is invalid, and print explanation.
   Return -2 if memory allocation failed, and print explanation. */
int add_udp_with_route(void *ntwrk, int datapath[], int datapathlen,
		       double srcrate, double starttime, int datasize);

/* Add data to app src appid; sent at end of current timestep or later.
   - appid: id of the app src (udpsrc or tcpsrc).
   - datasize (bytes): amount of data being added.
   Return total amount of data to be sent on success.
   Return -1 if parameter is invalid, and print explanation. */
int add_appdata(void *ntwrk, int appid, int datasize);

/* Network update functions _____________________________________________*/

/* Initialize the random number generator with seed. */
void set_randomseed(void *ntwrk, int seed);

/* Update the state of the network over one timestep.
   Return the simulated time at the end of the timestep. */
double update_network(void *ntwrk);

/* Metric gettors _______________________________________________________*/

/* Link metrics */

/* Return number of bytes in link lnkid of network *ntwrk at
   the end of the current timestep.
   Return -1 if lnkid is invalid, and print explanation. */
int get_inbuffsize(void *ntwrk, int lnkid);

/* Return number of bytes that arrived at link lnkid of network *ntwrk
   over the current timestep.
   Return -1 if lnkid is invalid, and print explanation. */
double get_arrivedsize(void *ntwrk, int lnkid);

/* Return number of bytes that were dropped (due to buffer overflow)
   at link lnkid of network *ntwrk over the current timestep.
   Return -1 if lnkid is invalid, and print explanation. */
double get_droppedsize(void *ntwrk, int lnkid);

/* App metrics */

/* Return current value of sentsn for src appid in network *ntwrk
   (sentsn: highest data sequence number + 1 ever sent by the src).
   Return -1 if app appid does not exist or is not a (tcp or udp) src,
   and print explanation. */
int get_sentsn(void *ntwrk, int appid);

/* Return current value of ackdsn for tcpsrc appid in network *ntwrk
   (ackdsn: highest ack sequence number received by tcpsrc).
   Return -1 if app appid does not exist or is not a tcpsrc,
   and print explanation. */
int get_ackdsn(void *ntwrk, int aid);

/* Return current value of cwsize for tcpsrc appid in network *ntwrk
   (cwsize: size (in bytes) of the congestion window at tcpsrc).
   Return -1 if app appid does not exist or is not a tcpsrc,
   and print explanation. */
int get_cwsize(void *ntwrk, int aid);

/* Return current value of sentsize for src appid in network *ntwrk
   (sentsize: number of data bytes sent by src since the start).
   Return -1 if app appid does not exist or is not a (tcp or udp) src
   and print explanation. */
int get_sentsize(void *ntwrk, int appid);

/* Return current value of rcvdsn for snk appid in network *ntwrk
   (rcvdsn for tcpsnk: data sequence number next expected by tcpsnk + 1)
   (rcvdsn for udpsnk: highest data sequence number received by tcpsnk + 1)
   Return -1 if app appid does not exist or is not a (tcp or udp) snk,
   and print explanation. */
int get_rcvdsn(void *ntwrk, int aid);

/* Return current value of rcvdsize for snk appid in network *ntwrk
   (rcvdsize: number of data bytes received by snk since the start).
   Return -1 if app appid does not exist or is not a (tcp or udp) snk,
   and print explanation. */
int get_rcvdsize(void *ntwrk, int aid);

/* End of tssnet_0.h __________________________________________________*/
