// float_examples.c: Show binary layout of a float value in IEEE-754
// format for Normalized, Special, and Denormalized numbers
#include <stdio.h>
#include <math.h>   // macros for infinity

#define FLOAT_BITS 32  // sizeof(float)*8

void showbits(void *xp);
char *bitstr(int x, int bits);
char *bitstr_index(int bits);
char *bitstrp(void *x, int bits);

int main(){
  float fl = 0.0;               // floating point value
  int *fp = (int *) &fl;        // pointer to float memory
  *fp = 0xC378C000;             // assign specific bits to float
  // *fp = 0b11000011011110001100000000000000;
  //         |   |   |   |   |   |   |   |   
  printf("float: %.4f\n",fl);
  printf("bits:  %s\n",  bitstr(*fp, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  printf("Assigning 0xC378C000 to float\n");
  fl = 0xC378C000;              // compiler converts int constatnt to
  printf("float: %.4f\n",fl);   // float which produces a large value
  printf("bits:  %s\n",  bitstr(*fp, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  // Create value 1.0 with bits
  // 0b0 01111111 000 0000 0000 0000 0000 0000;
  //   s:exponent:mantissa (implied leading 1)
  printf("1.0 as a Float\n");
  *fp = 0b00111111100000000000000000000000;
  printf("float: %.4f %.4e\n",fl, fl);
  printf("bits:  %s\n",  bitstr(*fp, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  // Special values: exponent bits are all 1s 

  // Playing with infinity
  float pinf =  INFINITY;
  float ninf = -INFINITY;
  float odz  = 1.0 / 0.0;
  float nodz = -1.0 / 0.0;
  printf("Infinities: %f %f %f %f\n",pinf,ninf,odz,nodz);
  printf("pinf:  %s\n",  bitstrp(&pinf, FLOAT_BITS));
  printf("ninf:  %s\n",  bitstrp(&ninf, FLOAT_BITS));
  printf("odz:   %s\n",  bitstrp(&odz, FLOAT_BITS));
  printf("nodz:  %s\n",  bitstrp(&nodz, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  // NaN: not a number
  float nan = NAN;
  float odo = 0.0 / 0.0;
  float imi = INFINITY - INFINITY;
  printf("NANs: %f %f %f\n",nan,odo,imi);
  printf("nan:   %s\n",  bitstrp(&nan, FLOAT_BITS));
  printf("odo:   %s\n",  bitstrp(&odo, FLOAT_BITS));
  printf("imi:   %s\n",  bitstrp(&imi, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));
  
  printf("DENORMALIZED FLOATS\n"); 

  // Denormalized: very small numbers, exponent bits are all 0s
  float sm = 3.0;               // 1.1_2 * 2^1
  int exp_bits = 128;           // bits in float exponent: 1000_0000 = 128
  printf("exp:   %d\n",exp_bits);
  printf("sm:    %e\n",sm);
  printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS));
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  printf("Divde 3.0 by 2 repeated (125 times)\n");

  for(int i=0; i<125; i++){     
    sm /= 2.0;                  // divide by 2 125 times
    exp_bits--;
  }
  printf("--CLOSE TO DENORMALIZED---\n");
  printf("exp:   %d\n",exp_bits);                   // 3
  printf("sm:    %e\n",sm);                         // 1.1 * 2^-124 =~= 7.052966e-38
  printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS)); // 0_00000011_10000000000000000000000
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  sm /= 2.0;
  exp_bits--;
  printf("exp:   %d\n",exp_bits);                   // 2
  printf("sm:    %e\n",sm);                         // 1.1 * 2^-125 =~= 3.526483e-38
  printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS)); // 0_00000010_10000000000000000000000
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  sm /= 2.0;
  exp_bits--;
  printf("exp:   %d\n",exp_bits);                   // 1
  printf("sm:    %e\n",sm);                         // 1.1 * 2^-126 =~= 1.763242e-38
  printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS)); // 0_00000010_10000000000000000000000
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));


  printf("---DENORMALIZED---\n");
  sm /= 2.0;
  exp_bits--;
  printf("exp:   %d\n",exp_bits);                   // 0
  printf("sm:    %e\n",sm);                         // 3 * 2^-126 =~= 1.763242e-38
  printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS)); // 
  printf("index: %s\n\n",bitstr_index(FLOAT_BITS));

  for(int i=0; i < 25; i++){
     sm /= 2.0;
     printf("sm:    %e\n",sm);
     printf("bits:  %s\n",  bitstrp(&sm, FLOAT_BITS));
     printf("index: %s\n\n",bitstr_index(FLOAT_BITS));
  }
  return 0;
}

#define INT_BITS 32


// Bit clusters for 32-bit float in IEEE-754 format
int clusters[] = {
  1, 8, 7, 8, 8,
};
int NCLUSTERS = sizeof(clusters)/sizeof(int);

// Generate a string version of the bits which clusters the bits based
// on the logical groupings in the problem
char *bitstr(int x, int bits){
  static char buffer[512];
  int idx = 0;
  int clust_idx = 0;
  int clust_break = clusters[0];
  for(int i=0; i<bits; i++){
    if(i==clust_break){
      buffer[idx] = ' ';
      idx++;
      clust_idx++;
      clust_break += clusters[clust_idx];
    }
    buffer[idx] = x & (1 << (31-i)) ? '1' : '0';
    idx++;
  }
  buffer[idx] = '\0';
  return buffer;
}

// Creates a string of bit indices matching the clustering pattern
// above
char *bitstr_index(int bits){
  static char buffer[512];
  char format[256];
  int pos = 0;
  int idx = bits;
  for(int i=0; i<NCLUSTERS; i++){
    idx -= clusters[i];
    if(clusters[i] > 1){
      sprintf(format, "%s%dd ","%",clusters[i]);
      pos += sprintf(buffer+pos, format, idx);
    }
    else{                       // cluster of 1 bit gets no index
      pos += sprintf(buffer+pos, "  ");
    }
  }
  buffer[pos-1] = '\0';         // eliminate trailing space
  return buffer;
}

// Run bitstr on a pointer to an integer
char *bitstrp(void *x, int bits){
  return bitstr(*(long *) x, bits);
}

/* Print the most signficant (right-most) to least signficant bit in
   the integer passed in */
void showbits(void *xp){
  int x = *((int *) xp);
  int mask = 0x1;
  for(int i=INT_BITS-1; i>=0; i--){
    int shifted_mask = mask << i;
    if(shifted_mask & x){
      printf("1");
    } else {
      printf("0");
    }
  }
  printf("\n");
}
